// Training types mapping
const trainingTypes = {
  yoga: { name: 'Йога', trainer: 'Анна Петрова' },
  crossfit: { name: 'Кроссфит', trainer: 'Иван Сидоров' },
  pilates: { name: 'Пилатес', trainer: 'Мария Козлова' },
  stretching: { name: 'Стретчинг', trainer: 'Ольга Новикова' }
};

// Initialize default bookings if not exists
function initDefaultBookings() {
  const saved = localStorage.getItem('fitnessBookings');
  if (!saved) {
    const defaultBookings = {
      active: [
        {
          id: 1,
          type: 'Йога',
          date: '2025-03-20',
          time: '18:00',
          trainer: 'Анна Петрова',
          status: 'active'
        },
        {
          id: 2,
          type: 'Кроссфит',
          date: '2025-03-22',
          time: '19:30',
          trainer: 'Иван Сидоров',
          status: 'active'
        },
        {
          id: 3,
          type: 'Пилатес',
          date: '2025-03-24',
          time: '17:00',
          trainer: 'Мария Козлова',
          status: 'active'
        }
      ],
      past: [
        {
          id: 4,
          type: 'Йога',
          date: '2025-03-15',
          time: '18:00',
          trainer: 'Анна Петрова',
          rating: 5
        },
        {
          id: 5,
          type: 'Стретчинг',
          date: '2025-03-10',
          time: '20:00',
          trainer: 'Ольга Новикова',
          rating: 4
        },
        {
          id: 6,
          type: 'Кроссфит',
          date: '2025-03-08',
          time: '19:30',
          trainer: 'Иван Сидоров',
          rating: 5
        }
      ]
    };
    localStorage.setItem('fitnessBookings', JSON.stringify(defaultBookings));
  }
}

// Get bookings from localStorage
function getBookings() {
  const saved = localStorage.getItem('fitnessBookings');
  return saved ? JSON.parse(saved) : { active: [], past: [] };
}

// Save bookings to localStorage
function saveBookings(bookings) {
  localStorage.setItem('fitnessBookings', JSON.stringify(bookings));
}

// Format date
function formatDate(dateString) {
  const date = new Date(dateString);
  return date.toLocaleDateString('ru-RU', { day: 'numeric', month: 'long' });
}

// Initialize page based on current page
function initPage() {
  const currentPage = window.location.pathname.split('/').pop();
  
  if (currentPage === 'index.html' || currentPage === '' || currentPage.includes('index')) {
    initBookingsPage();
  } else if (currentPage === 'book.html' || currentPage.includes('book')) {
    initBookingForm();
  }
}

// Initialize bookings page
function initBookingsPage() {
  initDefaultBookings();
  const bookings = getBookings();
  
  // Update stats
  updateStats(bookings);
  
  // Initialize tabs
  const tabs = document.querySelectorAll('.tab');
  let activeTab = 'active';
  
  tabs.forEach(tab => {
    tab.addEventListener('click', () => {
      tabs.forEach(t => t.classList.remove('active'));
      tab.classList.add('active');
      activeTab = tab.dataset.tab;
      displayBookings(bookings, activeTab);
    });
  });
  
  // Display initial bookings
  displayBookings(bookings, activeTab);
  
  // Update bookings periodically
  setInterval(() => {
    const updatedBookings = getBookings();
    updateStats(updatedBookings);
    displayBookings(updatedBookings, activeTab);
  }, 500);
}

// Update statistics
function updateStats(bookings) {
  const activeCount = document.getElementById('activeCount');
  const pastCount = document.getElementById('pastCount');
  const avgRating = document.getElementById('avgRating');
  
  if (activeCount) {
    activeCount.textContent = bookings.active.length;
  }
  
  if (pastCount) {
    pastCount.textContent = bookings.past.length;
  }
  
  if (avgRating) {
    const ratings = bookings.past.filter(b => b.rating).map(b => b.rating);
    const avg = ratings.length > 0 
      ? (ratings.reduce((a, b) => a + b, 0) / ratings.length).toFixed(1)
      : '0';
    avgRating.textContent = avg;
  }
}

// Display bookings
function displayBookings(bookings, tab) {
  const container = document.getElementById('bookingsContainer');
  if (!container) return;
  
  const bookingsList = tab === 'active' ? bookings.active : bookings.past;
  
  if (bookingsList.length === 0) {
    container.innerHTML = `
      <div class="empty-state">
        <p>Нет ${tab === 'active' ? 'активных' : 'прошедших'} записей</p>
      </div>
    `;
    return;
  }
  
  container.innerHTML = bookingsList.map(booking => {
    if (tab === 'active') {
      return `
        <div class="booking-card">
          <div class="booking-header">
            <h3 class="booking-type">${booking.type}</h3>
            <span class="status-badge">Активна</span>
          </div>
          <div class="booking-details">
            <div class="detail-item">${formatDate(booking.date)}</div>
            <div class="detail-item">${booking.time}</div>
            <div class="detail-item">${booking.trainer}</div>
          </div>
        </div>
      `;
    } else {
      return `
        <div class="booking-card past">
          <div class="booking-header">
            <h3 class="booking-type">${booking.type}</h3>
            <span class="status-badge past">Завершена</span>
          </div>
          <div class="booking-details">
            <div class="detail-item">${formatDate(booking.date)}</div>
            <div class="detail-item">${booking.time}</div>
            <div class="detail-item">${booking.trainer}</div>
          </div>
          <div class="rating-section">
            <p class="rating-label">Оцените тренера:</p>
            <div class="rating-stars">
              ${[1, 2, 3, 4, 5].map(star => `
                <button 
                  class="star-btn ${booking.rating >= star ? 'filled' : ''}"
                  onclick="rateTrainer(${booking.id}, ${star})"
                >
                  ⭐
                </button>
              `).join('')}
            </div>
          </div>
        </div>
      `;
    }
  }).join('');
}

// Rate trainer
function rateTrainer(bookingId, rating) {
  const bookings = getBookings();
  const booking = bookings.past.find(b => b.id === bookingId);
  if (booking) {
    booking.rating = rating;
    saveBookings(bookings);
    displayBookings(bookings, 'past');
    updateStats(bookings);
  }
}

// Initialize booking form
function initBookingForm() {
  initDefaultBookings();
  
  let selectedType = null;
  let selectedDate = null;
  let selectedTime = null;
  
  // Initialize dates
  const datesGrid = document.getElementById('datesGrid');
  if (datesGrid) {
    const dates = [];
    for (let i = 0; i < 7; i++) {
      const date = new Date();
      date.setDate(date.getDate() + i);
      dates.push(date.toISOString().split('T')[0]);
    }
    
    datesGrid.innerHTML = dates.map(date => {
      const dateObj = new Date(date);
      return `
        <button type="button" class="date-card" data-date="${date}">
          <div class="date-day">${dateObj.getDate()}</div>
          <div class="date-month">${dateObj.toLocaleDateString('ru-RU', { month: 'short' })}</div>
        </button>
      `;
    }).join('');
    
    datesGrid.querySelectorAll('.date-card').forEach(card => {
      card.addEventListener('click', () => {
        datesGrid.querySelectorAll('.date-card').forEach(c => c.classList.remove('selected'));
        card.classList.add('selected');
        selectedDate = card.dataset.date;
        updateSubmitButton();
      });
    });
  }
  
  // Initialize times
  const timesGrid = document.getElementById('timesGrid');
  if (timesGrid) {
    const times = ['09:00', '10:00', '11:00', '12:00', '17:00', '18:00', '19:00', '20:00'];
    timesGrid.innerHTML = times.map(time => `
      <button type="button" class="time-card" data-time="${time}">${time}</button>
    `).join('');
    
    timesGrid.querySelectorAll('.time-card').forEach(card => {
      card.addEventListener('click', () => {
        timesGrid.querySelectorAll('.time-card').forEach(c => c.classList.remove('selected'));
        card.classList.add('selected');
        selectedTime = card.dataset.time;
        updateSubmitButton();
      });
    });
  }
  
  // Initialize training types
  const trainingTypesContainer = document.getElementById('trainingTypes');
  if (trainingTypesContainer) {
    trainingTypesContainer.querySelectorAll('.type-card').forEach(card => {
      card.addEventListener('click', () => {
        trainingTypesContainer.querySelectorAll('.type-card').forEach(c => c.classList.remove('selected'));
        card.classList.add('selected');
        selectedType = card.dataset.type;
        updateSubmitButton();
      });
    });
  }
  
  // Update submit button state
  function updateSubmitButton() {
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
      submitBtn.disabled = !(selectedType && selectedDate && selectedTime);
    }
  }
  
  // Handle form submission
  const form = document.getElementById('bookingForm');
  if (form) {
    form.addEventListener('submit', (e) => {
      e.preventDefault();
      
      if (selectedType && selectedDate && selectedTime) {
        const bookings = getBookings();
        const typeInfo = trainingTypes[selectedType];
        
        const newBooking = {
          id: Date.now(),
          type: typeInfo.name,
          date: selectedDate,
          time: selectedTime,
          trainer: typeInfo.trainer,
          status: 'active'
        };
        
        bookings.active.push(newBooking);
        saveBookings(bookings);
        
        // Reset form
        selectedType = null;
        selectedDate = null;
        selectedTime = null;
        
        trainingTypesContainer.querySelectorAll('.type-card').forEach(c => c.classList.remove('selected'));
        datesGrid.querySelectorAll('.date-card').forEach(c => c.classList.remove('selected'));
        timesGrid.querySelectorAll('.time-card').forEach(c => c.classList.remove('selected'));
        updateSubmitButton();
        
        // Redirect to bookings page
        window.location.href = 'index.html';
      }
    });
  }
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initPage);
} else {
  initPage();
}

